import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getSalesCollection } from "@/lib/database/collections"
import { calculateSalesTotals } from "@/lib/utils/calculations"

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const salesCollection = await getSalesCollection()
    const sale = await salesCollection.findOne({ _id: new ObjectId(params.id) })

    if (!sale) {
      return NextResponse.json({ success: false, error: "Sales record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, data: sale })
  } catch (error) {
    console.error("Error fetching sale:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch sales record" }, { status: 500 })
  }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const body = await request.json()
    const totals = calculateSalesTotals(body)

    const updateData = {
      ...body,
      date: new Date(body.date),
      branchId: new ObjectId(body.branchId),
      totalCash: totals.totalCash,
      totalCard: totals.totalCard,
      totalSales: totals.totalSales,
      updatedAt: new Date(),
    }

    const salesCollection = await getSalesCollection()
    const result = await salesCollection.updateOne({ _id: new ObjectId(params.id) }, { $set: updateData })

    if (result.matchedCount === 0) {
      return NextResponse.json({ success: false, error: "Sales record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, data: updateData })
  } catch (error) {
    console.error("Error updating sale:", error)
    return NextResponse.json({ success: false, error: "Failed to update sales record" }, { status: 500 })
  }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const salesCollection = await getSalesCollection()
    const result = await salesCollection.deleteOne({ _id: new ObjectId(params.id) })

    if (result.deletedCount === 0) {
      return NextResponse.json({ success: false, error: "Sales record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, message: "Sales record deleted" })
  } catch (error) {
    console.error("Error deleting sale:", error)
    return NextResponse.json({ success: false, error: "Failed to delete sales record" }, { status: 500 })
  }
}
